﻿using Hl7.Fhir.Model;
using Hl7.Fhir.Serialization;
using log4net;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Http;
using System.Net.Http.Formatting;
using System.Net.Http.Headers;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.CreateResponseService.Interface;
using VA.PPMS.IWS.Functions.Configuration.Interface;
using VA.PPMS.IWS.QueueService.Interface;
using VA.PPMS.ProviderData;
using System.ServiceModel;

namespace VA.PPMS.IWS.CreateResponseService
{
    public class CreateResponseService : ICreateResponseService
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;
        private readonly IQueueService _queueService;
        private readonly IPpmsHelper _ppmsHelper;
        private readonly IPpmsContextHelper _contextHelper;

        public CreateResponseService(ILog logger, IIwsConfiguration configuration, IQueueService queueService, IPpmsHelper ppmsHelper, IPpmsContextHelper contextHelper)
        {
            _logger = logger;
            _configuration = configuration;
            _queueService = queueService;
            _ppmsHelper = ppmsHelper;
            _contextHelper = contextHelper;
        }

        public async Task<string> CreateResponse(DasMessage message)
        {
            _logger.Info($"@@@@ INFO - Start CreateResponseService for TransactionId: { message.TransactionId} @@@@");

            try
            {
                var batchId = message.Content;
                if (string.IsNullOrEmpty(batchId))
                {
                    _logger.Info($"@@@@ INFO - Batch ID not provided for TransactionId: {message.TransactionId} @@@@");
                    throw new PpmsServiceException($"CreateResponseService - Batch ID not provided for TransactionId {message.TransactionId}.");
                }

                // Create XML response file
                var response = await CreateResponseDocument(message);

                _logger.Info($"@@@@ INFO - End Main Create Response Service for TransactionId: {message.TransactionId} BatchId: {batchId} @@@@");

                return response;
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was a problem with theCreateResponseService for TransactionId {message.TransactionId} @@@@", ex);
                throw new PpmsServiceException($"There was a problem with the Main Orchestration Service for TransactionId {message.TransactionId}.", ex);
            }
        }

        public async Task<string> CreateProviderPayload(DasMessage message)
        {
            _logger.Info($"@@@@ INFO - Start CreateResponseService for TransactionId: { message.TransactionId} @@@@");

            try
            {
                // Create XML response file
                var response = await CreateProviderXmlDoc(message);

                _logger.Info($"@@@@ INFO - End Main Create Response Service for TransactionId: {message.TransactionId} BatchId: {message.Content} @@@@");

                return response;
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was a problem with theCreateResponseService for TransactionId {message.TransactionId} @@@@", ex);
                throw new PpmsServiceException($"There was a problem with the Main Orchestration Service for TransactionId {message.TransactionId}.", ex);
            }
        }

        public async Task NotifyOfResponse(DasMessage message)
        {
            var batchId = message.Content;

            try
            {
                var batch = await GetBatch(batchId);
                if (batch == null) throw new PpmsServiceException("Unable to find batch record.");

                // Retrieve ReceiverId from associated network
                if (batch.ppms_vaprovidernetwork_batch_network == null) throw new PpmsServiceException("Unable to determine associated network.");

                // Set header values
                message.ConversationId = batch.ppms_conversationid;
                message.ReceiverId = batch.ppms_vaprovidernetwork_batch_network.ppms_shorthand;

                var documentRef = await CreateDocumentReference(batchId, batch.ppms_conversationid);
                
                // Get URL paths from configuration
                var baseUrl = await _configuration.GetPpmsResponseNotificationUriAsync();
                var requestUri = await _configuration.GetPpmsResponsePostUriAsync();

                _logger.Info($"Posting to DAS at URL {baseUrl}{requestUri}");
                _logger.Info($"---Message: 1:{message.ConversationId}, 2:{message.SenderId}, 3:{message.ReceiverId}, 4:{message.SenderId}, 5:{message.Content}");

                var result = await PostToDas(message, documentRef, baseUrl, requestUri);

                _logger.Info($"Updating batch: {batchId}");

                if (!string.IsNullOrEmpty(result))
                {
                    await _ppmsHelper.UpdateBatch(message, $"Response notification sent: {batchId}", (int)ppms_batch_StatusCode.ReceiverNotified);
                }
            }
            catch (Exception ex)
            {
                _logger.Error("CreateResponseService.NotifyOfResponse: Unable to process response.");
                throw new PpmsServiceException("CreateResponseService.NotifyOfResponse: Unable to process response.", ex);
            }
        }

        private async Task<string> PostToDas(DasMessage message, DocumentReference content, string baseUri, string requestUri)
        {
            using (var client = new HttpClient())
            {
                client.BaseAddress = new Uri(baseUri);
                // Set DAS headers
                client.DefaultRequestHeaders.Accept.Clear();
                client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json+fhir"));
                client.DefaultRequestHeaders.Add("X-ConversationID", message.ConversationId);
                client.DefaultRequestHeaders.Add("X-RoutingSenderID", message.SenderId);
                client.DefaultRequestHeaders.Add("X-RoutingReceiverIDs", message.ReceiverId);
                client.DefaultRequestHeaders.Add("X-TransactionID", message.TransactionId);

                var response = await client.PostAsync(requestUri, new ObjectContent(typeof(DocumentReference), content, new NoCharSetJsonMediaTypeFormatter(), "application/json+fhir"));

                if (response.IsSuccessStatusCode) return "Success";

                throw new PpmsServiceException($"Error Posting Response Notification to DAS for transaction {message.TransactionId}. The error is {response.StatusCode.ToString()}");
            }
        }

        private async Task<DocumentReference> CreateDocumentReference(string batchId, string conversationId)
        {
            // Construct call back URL
            //var referencePath = await _configuration.GetResponseDocumentReferenceUrlAsync();
            var documentPath = await GetDocumentPath(batchId, conversationId);
            var content = new List<DocumentReference.ContentComponent>
            {
                new DocumentReference.ContentComponent { Attachment = new Attachment { ContentType = "application/xml", Url = $"{documentPath}" } }
            };

            // Set DocumentReference properties
            var docRef = new DocumentReference
            {
                Custodian = new ResourceReference("PPMS"),
                Created = DateTime.Now.ToString("s", System.Globalization.CultureInfo.InvariantCulture),
                Indexed = new DateTimeOffset(DateTime.Now),
                Status = DocumentReferenceStatus.Current,
                Content = content
            };

            return docRef;
        }

        private async Task<string> GetDocumentPath(string batchId, string conversationId)
        {
            if (string.IsNullOrEmpty(batchId) || string.IsNullOrEmpty(conversationId))
            {
                _logger.Info($"@@@@ INFO - Parameter invalid for BatchId: {batchId}, ConversationId {conversationId} @@@@");
                throw new PpmsServiceException("Unable to create document path, parameter invalid");
            }

            //{ relative - path}/ Binary /[documentURN] ? transactionID ={ unique ID}
            //&requestor ={ ES | PPMS | CUI etc.}
            //&purposeOfUse = Treatment & dataSource ={ ES | PPMS | CUI etc.}
            var relativePathPattern = await _configuration.GetResponseDocumentPathPatternAsync();
            return string.Format(relativePathPattern, batchId, conversationId);
        }

        private async Task<ppms_batch> GetBatch(string batchId)
        {
            var context = await _contextHelper.GetContextAsync();
            var batch = context.ppms_batchSet.FirstOrDefault(b => b.ppms_batchId == new Guid(batchId));
            if (batch == null) throw new PpmsServiceException("Batch record does not exist");

            context.LoadProperty(batch, new Relationship("ppms_batch_batchdetail_batch"));
            context.LoadProperty(batch, new Relationship("ppms_vaprovidernetwork_batch_network"));

            if (batch.ppms_batch_batchdetail_batch != null)
            {
                foreach (var item in batch.ppms_batch_batchdetail_batch)
                {
                    context.LoadProperty(item, new Relationship("ppms_batchdetail_batchdetailresult"));
                }
            }

            return batch;
        }

        private async Task<IList<ppms_providernetworkid>> GetProviderNetworks(string providerId)
        {
            var context = await _contextHelper.GetContextAsync();
            var provider = context.ppms_providernetworkidSet.FirstOrDefault(b => b.ppms_providerid == providerId);
            if (provider == null) throw new PpmsServiceException("Provider record does not exist");

            //context.LoadProperty(provider, new Relationship("ppms_batch_batchdetail_batch"));

            //if (provider.ppms_batch_batchdetail_batch != null)
            //{
            //    foreach (var item in provider.ppms_batch_batchdetail_batch)
            //    {
            //        context.LoadProperty(item, new Relationship("ppms_batchdetail_batchdetailresult"));
            //    }
            //}

            return new List<ppms_providernetworkid>() { provider };
        }

        public async Task<string> CreateResponseDocument(DasMessage queueMessage)
        {
            try
            {
                if (string.IsNullOrEmpty(queueMessage.Content))
                {
                    _logger.Info($"@@@@ INFO - Parameter invalid for item: {queueMessage.ToString()} @@@@");
                    return string.Empty;
                }

                var batchId = queueMessage.Content;
                var batch = await GetBatch(batchId);

                _logger.Info($"@@@@ INFO - Verify details for ConversationId: {batchId} @@@@");
                var details = batch.ppms_batch_batchdetail_batch;
                IEnumerable<ppms_batchdetail> ppmsBatchdetails = details as ppms_batchdetail[] ?? details.ToArray();
                if (!ppmsBatchdetails.Any())
                {
                    _logger.Info($"@@@@ INFO - Details not found for BatchId: {batchId} @@@@");
                    return string.Empty;
                }

                _logger.Info($"@@@@ INFO - Details found for BatchId: {batchId} @@@@");

                // Create XML doc
                var doc = new ProviderResponses
                {
                    ProviderResponse = new List<ProviderResponse>(),
                    TransactionId = batch.ppms_transactionid
                };

                // Capture batch details
                foreach (var detail in ppmsBatchdetails)
                {
                    // Provider node
                    var provider = new ProviderResponse
                    {
                        ProviderId = detail.ppms_providerid,
                        Success = detail.GetAttributeValue<bool>("ppms_isvalid")
                    };

                    // Set correlation id, if appropriate
                    if (detail.ppms_provider != null)
                    {
                        provider.CorrelationId = detail.ppms_provider.Id.ToString();
                    }

                    // Retrieve batch detail results
                    var batchDetailResults = detail.ppms_batchdetail_batchdetailresult;

                    // Capture batch detail results
                    IEnumerable<ppms_batchdetailresult> ppmsBatchdetailresults = batchDetailResults as ppms_batchdetailresult[] ?? batchDetailResults.ToArray();
                    if (ppmsBatchdetailresults.Any())
                    {
                        // Initialize results list
                        provider.Results = new Results { Item = new List<Result>() };

                        foreach (var detailResult in ppmsBatchdetailresults)
                        {
                            var result = new Result
                            {
                                Type = detailResult.ppms_entitytype,
                                Id = detailResult.ppms_name,
                                Success = detailResult.ppms_isvalid.HasValue && detailResult.ppms_isvalid.Value
                            };

                            if (!result.Success)
                            {
                                result.Header = detailResult.ppms_result;
                                result.Message = detailResult.ppms_message;
                            }

                            provider.Results.Item.Add(result);
                        }
                    }
                    else
                    {
                        _logger.Info($"@@@@ INFO - Results not found for provider: {detail.ppms_name} @@@@");
                    }

                    // Add to list of providers
                    doc.ProviderResponse.Add(provider);
                }

                // Create response packet
                var responseToSend = await ConvertResponseToXml(doc);

                // Close batch
                await _ppmsHelper.UpdateBatch(queueMessage, "CCN Response sent.", (int)ppms_batch_StatusCode.Complete);

                _logger.Info($"@@@@ INFO - Save complete for BatchId: {batchId} @@@@");

                return responseToSend;
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                _logger.Error($"@@@@ CreateResponseService ERROR - Fault: {ex} @@@@", ex);
                throw;
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ CreateResponseService ERROR - Exception: {ex} @@@@", ex);
                throw;
            }
        }

        public async Task<string> CreateProviderXmlDoc(DasMessage queueMessage)
        {
            try
            {
                if (string.IsNullOrEmpty(queueMessage.Content))
                {
                    _logger.Info($"@@@@ INFO - Batch ID not provided for item: {queueMessage.ToString()} @@@@");
                    return string.Empty;
                }

                var batchId = queueMessage.Content;
                var batch = await GetBatch(batchId);

                return await ExportBatchToXml(batch);
            }
            catch (Exception ex)
            {
                _logger.Info($"!!!! ERROR - Error occured created provider XML document: \n{queueMessage.ToString()}\n{ex.Message} !!!!");
            }

            return string.Empty;
        }

        private async Task<string> ConvertResponseToXml(ProviderResponses response)
        {
            var packet = await _configuration.GetSchemaProfileAsync(SchemaOptions.SchemaProfiles.Response);

            var prefix = packet.Prefix;
            var nameSpace = packet.Namespace;

            return Utilities.SerializeInstance(response, prefix, nameSpace);
        }

        private async Task<string> ExportBatchToXml(ppms_batch batch)
        {
            Context.Account account = null;

            if (batch.ppms_batch_batchdetail_batch == null) return string.Empty;

            var providers = new List<Provider>();

            foreach (var item in batch.ppms_batch_batchdetail_batch)
            {
                account = await GetProvider(item.ppms_provider.Id);
                if (account != null)
                {
                    // convert provider to XML
                    providers.Add(ToProvider(account));
                }
            }

            var xml = Utilities.SerializeInstance(providers, "p", "https://ppms.DNS   /exchange/ccn/1.0");

            return xml;
        }

        private async Task<Context.Account> GetProvider(Guid providerId)
        {
            var context = await _contextHelper.GetContextAsync();
            var entity = context.AccountSet.FirstOrDefault(b => b.Id == providerId);
            if (entity == null) throw new PpmsServiceException("Batch record does not exist");

            LoadProviderProperties(entity, context);

            return entity;
        }

        protected void LoadProviderProperties(Entity entity, PpmsContext context)
        {
            if (entity == null || context == null) return;

            context.LoadProperty(entity, "ppms_account_ppms_provideridentifier_Provider");
            context.LoadProperty(entity, "ppms_account_ppms_providerservice");
            context.LoadProperty(entity, "ppms_account_ppms_boardcertification");
            context.LoadProperty(entity, "ppms_account_organizationauthorizedofficial");
            context.LoadProperty(entity, "ppms_account_ppms_otherprovideridentifier");
            context.LoadProperty(entity, "contact_customer_accounts");
            context.LoadProperty(entity, "ppms_account_providerlicensure");
            context.LoadProperty(entity, "ppms_account_ppms_othername");
            context.LoadProperty(entity, "ppms_account_ppms_providertaxonomy");
            context.LoadProperty(entity, "ppms_account_deascheduleprivilege");
            context.LoadProperty(entity, "ppms_account_providernetworkid");
        }

        private Provider ToProvider(Context.Account account)
        {
            if (account == null) return null;

            var provider = new Provider();

            // Map properties
            provider.Email = account.EMailAddress1;
            provider.Phone = account.Telephone1;
            provider.Fax = account.Fax;

            // Address
            var address = new ProviderData.Address();
            address.Address1 = account.Address1_Line1;
            address.Address2 = account.Address1_Line2;
            address.Address3 = account.Address1_Line3;
            address.City = account.Address1_City;
            address.State = account.Address1_StateOrProvince;
            address.PostalCode = account.Address1_PostalCode;
            address.County = account.Address1_County;

            provider.Address = address;

            // NPIs
            var npis = new List<Npi>();
            var npi = new Npi();
            npi.Number = account.ppms_ProviderIdentifier;
            provider.Npis = new Npis() { Item = npis };

            // Specialties
            if (account.ppms_account_ppms_providertaxonomy != null)
            {
                var specialties = new List<Taxonomy>();
                foreach (var item in account.ppms_account_ppms_providertaxonomy)
                {
                    specialties.Add(new Taxonomy() { CodedSpecialty = item.ppms_codedspecialty });
                }
                provider.Specialties = new Specialties() { Item = specialties };
            }

            // Name
            var individual = new Individual();
            var names = account.Name.Split(',');
            individual.FirstName = names[0].Trim();
            individual.LastName = names[1].Trim();
            individual.MiddleName = "";

            // DEA Numbers
            if (account.ppms_account_deascheduleprivilege != null)
            {
                var deaNumbers = new List<DeaSchedulePrivilege>();
                foreach (var item in account.ppms_account_deascheduleprivilege)
                {
                    deaNumbers.Add(new DeaSchedulePrivilege() { DeaNumber = item.ppms_deanumber });
                }
                individual.DeaNumbers = new DeaSchedulePrivileges() { Item = deaNumbers };
            }

            // Set provider type
            var providerType = new ProviderType();
            providerType.Item = individual;

            provider.Type = providerType;

            return provider;
        }
    }
}